# services/openai_service.py
import os
from openai import OpenAI
from dotenv import load_dotenv

# Загрузка переменных окружения
load_dotenv()


class OpenAIService:
    def __init__(self):
        # Получение API ключа при инициализации
        self.api_key = os.getenv("OPENAI_API_KEY")
        if not self.api_key:
            raise ValueError("OPENAI_API_KEY не найден в переменных окружения")

        # Инициализация клиента OpenAI
        self.client = OpenAI(api_key=self.api_key)

    def transcribe_audio(self, audio_file_path, output_file_path="files/transcription.txt"):
        """
        Транскрибирует аудиофайл используя OpenAI Whisper API

        Args:
            audio_file_path: Путь к аудиофайлу
            output_file_path: Путь для сохранения транскрипции

        Returns:
            str: Текст транскрипции
        """
        # Проверка существования аудиофайла
        if not os.path.exists(audio_file_path):
            raise FileNotFoundError(f"Файл {audio_file_path} не найден")

        try:
            with open(audio_file_path, "rb") as audio_file:
                print(f"Транскрибация файла: {audio_file_path}")

                # Вызов Whisper API
                transcription = self.client.audio.transcriptions.create(
                    model="whisper-1",
                    file=audio_file,
                    response_format="text"
                )

            # Создаем директорию если её нет
            os.makedirs(os.path.dirname(output_file_path), exist_ok=True)

            with open(output_file_path, "w", encoding="utf-8") as output_file:
                output_file.write(transcription)

            return transcription

        except Exception as e:
            print(f"Ошибка при транскрибации: {str(e)}")
            raise e

    def process_text(self, text: str, prompt: str = "", model: str = "gpt-4o", temperature: float = 0.7):
        """
        Обрабатывает текст через OpenAI API

        Args:
            text: Исходный текст для обработки
            prompt: Служебный промпт
            model: Модель для использования

        Returns:
            str: Ответ от модели
        """
        try:
            # Формирование сообщений для модели
            messages = []
            if prompt:
                messages.append({"role": "system", "content": prompt})
            messages.append({"role": "user", "content": text})

            # Отправка запроса к OpenAI API
            response = self.client.chat.completions.create(
                model=model,
                messages=messages,
                temperature=temperature
            )

            return response.choices[0].message.content

        except Exception as e:
            print(f"Ошибка при обработке текста через OpenAI: {str(e)}")
            raise e