#!/usr/bin/env python3
# main.py - Транскрибация аудио файла с помощью OpenAI Whisper API

import os
import sys
from openai import OpenAI
from dotenv import load_dotenv

# Загрузка переменных окружения
load_dotenv()


def transcribe_audio(audio_file_path, output_file_path="transcription.txt"):
    """
    Транскрибирует аудиофайл используя OpenAI Whisper API

    Args:
        audio_file_path: Путь к аудиофайлу
        output_file_path: Путь для сохранения транскрипции
    """

    # Получение API ключа из переменных окружения
    api_key = os.getenv("OPENAI_API_KEY")
    if not api_key:
        print("Ошибка: OPENAI_API_KEY не найден в переменных окружения")
        sys.exit(1)

    # Проверка существования аудиофайла
    if not os.path.exists(audio_file_path):
        print(f"Ошибка: Файл {audio_file_path} не найден")
        sys.exit(1)

    try:
        # Инициализация клиента OpenAI
        client = OpenAI(api_key=api_key)

        # Открытие и отправка аудиофайла для транскрибации
        with open(audio_file_path, "rb") as audio_file:
            print(f"Транскрибация файла: {audio_file_path}")

            # Вызов Whisper API
            transcription = client.audio.transcriptions.create(
                model="whisper-1",
                file=audio_file,
                response_format="text"  # Можно также использовать "json", "srt", "vtt"
            )

        # Сохранение результата в файл
        with open(output_file_path, "w", encoding="utf-8") as output_file:
            output_file.write(transcription)

        print(f"Транскрипция успешно сохранена в: {output_file_path}")

        # Вывод первых 200 символов для проверки
        preview = transcription[:200] + "..." if len(transcription) > 200 else transcription
        print(f"\nПредпросмотр транскрипции:\n{preview}")

    except Exception as e:
        print(f"Ошибка при транскрибации: {str(e)}")
        sys.exit(1)


def main():
    # Путь к аудиофайлу
    audio_path = "audio.mp3"

    # Путь для сохранения транскрипции
    output_path = "transcription.txt"

    # Выполнение транскрибации
    transcribe_audio(audio_path, output_path)


if __name__ == "__main__":
    main()